# ESP32 Peripheral Manager API Documentation

## Overview

The ESP32 Peripheral Manager (PerimaN) is a centralized GPIO pin management system for ESP32 Arduino Core. It provides conflict prevention, resource tracking, and clean initialization/deinitialization patterns for all ESP32 peripherals.

## Features

- **Pin Conflict Prevention**: Automatically detects and prevents GPIO pin conflicts between peripherals
- **Resource Tracking**: Maintains state of all GPIO pins and their assigned peripherals  
- **Automatic Cleanup**: Provides callback mechanisms for proper peripheral deinitialization
- **Type Safety**: Strong typing for peripheral bus types and validation
- **Multi-peripheral Support**: Supports all ESP32 peripheral types (UART, I2C, SPI, LEDC, RMT, etc.)

## Core Data Structures

### peripheral_bus_type_t

Enumeration defining all supported peripheral bus types:

```c
typedef enum {
    ESP32_BUS_TYPE_INIT = 0,        ///< Initial/unassigned state
    ESP32_BUS_TYPE_GPIO,            ///< General Purpose I/O
    ESP32_BUS_TYPE_UART_RX,         ///< UART Receive pin
    ESP32_BUS_TYPE_UART_TX,         ///< UART Transmit pin
    ESP32_BUS_TYPE_UART_CTS,        ///< UART Clear to Send
    ESP32_BUS_TYPE_UART_RTS,        ///< UART Request to Send
    ESP32_BUS_TYPE_I2C_MASTER_SDA,  ///< I2C Data line (Master mode)
    ESP32_BUS_TYPE_I2C_MASTER_SCL,  ///< I2C Clock line (Master mode)
    ESP32_BUS_TYPE_LEDC,            ///< LED Controller PWM
    ESP32_BUS_TYPE_RMT_TX,          ///< Remote Control Transmit
    ESP32_BUS_TYPE_RMT_RX,          ///< Remote Control Receive
    ESP32_BUS_TYPE_SIGMADELTA,      ///< Sigma-Delta Modulator
    ESP32_BUS_TYPE_ADC_ONESHOT,     ///< ADC One-shot reading mode
    ESP32_BUS_TYPE_ADC_CONTINUOUS,  ///< ADC Continuous reading mode
    ESP32_BUS_TYPE_TOUCH,           ///< Touch sensor
    ESP32_BUS_TYPE_MAX              ///< Maximum value (boundary marker)
} peripheral_bus_type_t;
```

### peripheral_pin_item_t

Internal structure representing pin assignment information:

```c
typedef struct ATTR_PACKED {
    peripheral_bus_type_t type;     ///< Type of peripheral bus
    const char *extra_type;         ///< Additional type information
    void *bus;                      ///< Pointer to bus handle/context
    int8_t bus_num;                 ///< Bus number identifier
    int8_t bus_channel;             ///< Channel within the bus
} peripheral_pin_item_t;
```

### peripheral_bus_deinit_cb_t

Callback function type for peripheral deinitialization:

```c
typedef bool (*peripheral_bus_deinit_cb_t)(void *bus);
```

**Parameters:**
- `bus`: Pointer to the bus handle/context to deinitialize

**Returns:**
- `true`: Deinitialization successful
- `false`: Deinitialization failed

## API Functions

### perimanSetPinBus()

Assigns a GPIO pin to a specific peripheral bus.

```c
bool perimanSetPinBus(uint8_t pin, peripheral_bus_type_t type, 
                      void *bus, int8_t bus_num, int8_t bus_channel);
```

**Parameters:**
- `pin`: GPIO pin number (0 to SOC_GPIO_PIN_COUNT-1)
- `type`: Peripheral bus type from peripheral_bus_type_t enum
- `bus`: Pointer to peripheral bus handle/context (can be NULL)
- `bus_num`: Bus number identifier (-1 if not applicable)
- `bus_channel`: Channel number within the bus (-1 if not applicable)

**Returns:**
- `true`: Pin successfully assigned to peripheral
- `false`: Assignment failed (invalid pin, pin already in use, etc.)

**Error Conditions:**
- Invalid pin number (>= SOC_GPIO_PIN_COUNT)
- Pin not valid for GPIO operations
- Pin already assigned to another peripheral

**Usage Example:**
```c
// Assign pin 21 to I2C SDA bus 0
if (!perimanSetPinBus(21, ESP32_BUS_TYPE_I2C_MASTER_SDA, (void*)1, 0, -1)) {
    log_e("Failed to assign pin 21 to I2C SDA");
    return ESP_FAIL;
}
```

### perimanClearPinBus()

Releases a GPIO pin from its current peripheral assignment.

```c
bool perimanClearPinBus(uint8_t pin);
```

**Parameters:**
- `pin`: GPIO pin number to release

**Returns:**
- `true`: Pin successfully released
- `false`: Release failed or pin was not assigned

**Behavior:**
- If a deinit callback is registered for the peripheral type, it will be called
- Pin state is reset to ESP32_BUS_TYPE_INIT
- All associated bus information is cleared

**Usage Example:**
```c
// Release pin 21 from current assignment
if (!perimanClearPinBus(21)) {
    log_w("Pin 21 was not assigned or failed to release");
}
```

### perimanGetPinBus()

Retrieves the bus handle for a pin assigned to a specific peripheral type.

```c
void *perimanGetPinBus(uint8_t pin, peripheral_bus_type_t type);
```

**Parameters:**
- `pin`: GPIO pin number to query
- `type`: Expected peripheral bus type

**Returns:**
- Valid pointer: Bus handle if pin is assigned to the specified type
- `NULL`: Pin not assigned to the specified peripheral type

**Usage Example:**
```c
// Get I2C bus handle for pin 21
void *i2c_bus = perimanGetPinBus(21, ESP32_BUS_TYPE_I2C_MASTER_SDA);
if (i2c_bus != NULL) {
    // Pin is assigned to I2C SDA, use the bus handle
    // ...
}
```

### perimanGetPinBusType()

Gets the current peripheral bus type assignment for a GPIO pin.

```c
peripheral_bus_type_t perimanGetPinBusType(uint8_t pin);
```

**Parameters:**
- `pin`: GPIO pin number to query

**Returns:**
- Current peripheral bus type assigned to the pin
- `ESP32_BUS_TYPE_INIT` if pin is not assigned

**Usage Example:**
```c
peripheral_bus_type_t current_type = perimanGetPinBusType(21);
if (current_type != ESP32_BUS_TYPE_INIT) {
    log_i("Pin 21 is assigned to: %s", perimanGetTypeName(current_type));
}
```

### perimanGetTypeName()

Returns a human-readable string name for a peripheral bus type.

```c
const char *perimanGetTypeName(peripheral_bus_type_t type);
```

**Parameters:**
- `type`: Peripheral bus type enum value

**Returns:**
- Constant string representing the bus type name
- Never returns NULL (returns "UNKNOWN" for invalid types)

**Usage Example:**
```c
log_i("Pin 21 type: %s", perimanGetTypeName(ESP32_BUS_TYPE_I2C_MASTER_SDA));
// Output: "Pin 21 type: I2C_MASTER_SDA"
```

### perimanSetBusDeinit()

Registers a deinitialization callback for a specific peripheral bus type.

```c
void perimanSetBusDeinit(peripheral_bus_type_t type, peripheral_bus_deinit_cb_t cb);
```

**Parameters:**
- `type`: Peripheral bus type to register callback for
- `cb`: Callback function pointer (can be NULL to unregister)

**Behavior:**
- The callback will be invoked when `perimanClearPinBus()` is called
- Only one callback per bus type is supported
- Passing NULL removes the callback

**Usage Example:**
```c
static bool my_i2c_deinit(void *bus) {
    // Custom I2C deinitialization
    return i2c_driver_delete((i2c_port_t)(int)bus) == ESP_OK;
}

// Register the callback
perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SDA, my_i2c_deinit);
```

## Initialization and Usage Patterns

### Basic Initialization Sequence

```c
#include "esp32-hal-periman.h"

void setup_peripheral() {
    // 1. Register deinitialization callback
    perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SDA, i2c_deinit_callback);
    perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SCL, i2c_deinit_callback);
    
    // 2. Clear any existing pin assignments
    if (!perimanClearPinBus(SDA_PIN) || !perimanClearPinBus(SCL_PIN)) {
        log_e("Failed to clear pins");
        return;
    }
    
    // 3. Initialize your peripheral (I2C example)
    i2c_config_t conf = {
        .mode = I2C_MODE_MASTER,
        .sda_io_num = SDA_PIN,
        .scl_io_num = SCL_PIN,
        // ... other config
    };
    
    if (i2c_param_config(I2C_NUM_0, &conf) != ESP_OK) {
        log_e("I2C config failed");
        return;
    }
    
    if (i2c_driver_install(I2C_NUM_0, conf.mode, 0, 0, 0) != ESP_OK) {
        log_e("I2C driver install failed");
        return;
    }
    
    // 4. Register pins with peripheral manager
    if (!perimanSetPinBus(SDA_PIN, ESP32_BUS_TYPE_I2C_MASTER_SDA, 
                          (void*)(I2C_NUM_0 + 1), I2C_NUM_0, -1) ||
        !perimanSetPinBus(SCL_PIN, ESP32_BUS_TYPE_I2C_MASTER_SCL, 
                          (void*)(I2C_NUM_0 + 1), I2C_NUM_0, -1)) {
        log_e("Failed to register pins with peripheral manager");
        // Clean up
        i2c_driver_delete(I2C_NUM_0);
        return;
    }
    
    log_i("I2C peripheral successfully initialized");
}
```

### Pin Conflict Detection

```c
bool check_pin_available(uint8_t pin, peripheral_bus_type_t intended_type) {
    peripheral_bus_type_t current_type = perimanGetPinBusType(pin);
    
    if (current_type == ESP32_BUS_TYPE_INIT) {
        // Pin is available
        return true;
    }
    
    if (current_type == intended_type) {
        // Pin already assigned to same peripheral type
        log_w("Pin %d already assigned to %s", pin, perimanGetTypeName(current_type));
        return true;
    }
    
    // Pin conflict detected
    log_e("Pin %d conflict: currently %s, requested %s", 
          pin, perimanGetTypeName(current_type), perimanGetTypeName(intended_type));
    return false;
}
```

### Safe Peripheral Switching

```c
bool switch_pin_to_gpio(uint8_t pin) {
    peripheral_bus_type_t current_type = perimanGetPinBusType(pin);
    
    if (current_type == ESP32_BUS_TYPE_GPIO) {
        // Already GPIO
        return true;
    }
    
    if (current_type != ESP32_BUS_TYPE_INIT) {
        log_i("Switching pin %d from %s to GPIO", pin, perimanGetTypeName(current_type));
        
        // This will call the appropriate deinit callback
        if (!perimanClearPinBus(pin)) {
            log_e("Failed to clear pin %d", pin);
            return false;
        }
    }
    
    // Configure as GPIO
    gpio_config_t gpio_conf = {
        .pin_bit_mask = (1ULL << pin),
        .mode = GPIO_MODE_INPUT_OUTPUT,
        .pull_up_en = GPIO_PULLUP_DISABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_DISABLE
    };
    
    if (gpio_config(&gpio_conf) != ESP_OK) {
        log_e("GPIO config failed for pin %d", pin);
        return false;
    }
    
    // Register with peripheral manager
    if (!perimanSetPinBus(pin, ESP32_BUS_TYPE_GPIO, NULL, -1, -1)) {
        log_e("Failed to register pin %d as GPIO", pin);
        return false;
    }
    
    return true;
}
```

## Error Handling

### Common Error Conditions

1. **Invalid Pin Number**
   - **Condition**: Pin number >= SOC_GPIO_PIN_COUNT
   - **Error**: perimanSetPinBus() returns false
   - **Resolution**: Validate pin numbers before use

2. **Pin Not GPIO-Capable**
   - **Condition**: Pin doesn't support GPIO operations
   - **Error**: Pin assignment fails
   - **Resolution**: Check SOC_GPIO_VALID_GPIO_MASK

3. **Pin Already Assigned**
   - **Condition**: Attempting to assign already-used pin
   - **Error**: perimanSetPinBus() returns false
   - **Resolution**: Clear pin first or check availability

4. **Deinitialization Failure**
   - **Condition**: Registered deinit callback returns false
   - **Error**: perimanClearPinBus() returns false
   - **Resolution**: Handle partial cleanup scenarios

### Debugging Pin Assignments

```c
void debug_pin_assignments(void) {
    log_i("=== GPIO Pin Assignments ===");
    
    for (int pin = 0; pin < SOC_GPIO_PIN_COUNT; pin++) {
        peripheral_bus_type_t type = perimanGetPinBusType(pin);
        
        if (type != ESP32_BUS_TYPE_INIT) {
            void *bus = perimanGetPinBus(pin, type);
            log_i("Pin %2d: %-20s (bus: %p)", 
                  pin, perimanGetTypeName(type), bus);
        }
    }
    
    log_i("=== End Pin Assignments ===");
}
```

## Configuration Macros

### SOC-Specific Definitions

```c
// Maximum number of GPIO pins (varies by ESP32 variant)
#define SOC_GPIO_PIN_COUNT     // Defined in soc_caps.h

// Valid GPIO mask (some pins may be reserved)
#define SOC_GPIO_VALID_GPIO_MASK  // Defined in soc_caps.h

// Pin validation macro
#define GPIO_NOT_VALID(p) ((p >= SOC_GPIO_PIN_COUNT) || ((SOC_GPIO_VALID_GPIO_MASK & (1ULL << p)) == 0))
```

### Build Configuration

```c
// Attribute for packed structures
#define ATTR_PACKED __attribute__((packed))

// Maximum bus types supported
#define ESP32_BUS_TYPE_MAX  // Automatically calculated
```

## Best Practices

### 1. Always Register Deinit Callbacks

```c
// GOOD: Register deinit callback before any pin assignments
perimanSetBusDeinit(ESP32_BUS_TYPE_LEDC, ledc_deinit_callback);

// BAD: Missing deinit callback can cause resource leaks
perimanSetPinBus(pin, ESP32_BUS_TYPE_LEDC, bus, -1, -1);
```

### 2. Check Pin Availability

```c
// GOOD: Check before assignment
if (perimanGetPinBusType(pin) != ESP32_BUS_TYPE_INIT) {
    if (!perimanClearPinBus(pin)) {
        log_e("Failed to clear pin %d", pin);
        return ESP_FAIL;
    }
}

// BAD: Assuming pin is available
perimanSetPinBus(pin, ESP32_BUS_TYPE_GPIO, NULL, -1, -1);
```

### 3. Validate Return Values

```c
// GOOD: Always check return values
if (!perimanSetPinBus(pin, type, bus, num, channel)) {
    log_e("Pin assignment failed");
    // Handle error appropriately
    return ESP_FAIL;
}

// BAD: Ignoring return values
perimanSetPinBus(pin, type, bus, num, channel);
```

### 4. Use Consistent Bus Handles

```c
// GOOD: Use consistent, meaningful bus handles
#define I2C_BUS_HANDLE(port) ((void*)((port) + 1))

perimanSetPinBus(sda_pin, ESP32_BUS_TYPE_I2C_MASTER_SDA, 
                 I2C_BUS_HANDLE(i2c_port), i2c_port, -1);

// BAD: Inconsistent or NULL handles
perimanSetPinBus(sda_pin, ESP32_BUS_TYPE_I2C_MASTER_SDA, NULL, -1, -1);
```

## Integration Examples

### Arduino-style Peripheral Library

```c
class MyPeripheral {
private:
    uint8_t pin;
    peripheral_bus_type_t bus_type;
    void* bus_handle;
    
    static bool deinit_callback(void* bus) {
        // Cleanup peripheral resources
        return true;
    }
    
public:
    bool begin(uint8_t pin_num) {
        // Register deinit callback
        perimanSetBusDeinit(ESP32_BUS_TYPE_CUSTOM, deinit_callback);
        
        // Check pin availability
        if (perimanGetPinBusType(pin_num) != ESP32_BUS_TYPE_INIT) {
            if (!perimanClearPinBus(pin_num)) {
                return false;
            }
        }
        
        // Initialize peripheral hardware
        if (!init_hardware(pin_num)) {
            return false;
        }
        
        // Register with peripheral manager
        bus_handle = create_bus_handle();
        if (!perimanSetPinBus(pin_num, ESP32_BUS_TYPE_CUSTOM, 
                              bus_handle, -1, -1)) {
            cleanup_hardware();
            return false;
        }
        
        pin = pin_num;
        bus_type = ESP32_BUS_TYPE_CUSTOM;
        return true;
    }
    
    void end() {
        if (pin != 255) {
            perimanClearPinBus(pin);
            pin = 255;
        }
    }
};
```

## Troubleshooting Guide

### Issue: "Invalid pin: 255" Error

**Cause**: Using -1 or invalid pin numbers that get cast to 255 (uint8_t)

**Solution**:
```c
// BAD: Using -1 directly
perimanSetPinBus(-1, ESP32_BUS_TYPE_GPIO, NULL, -1, -1);

// GOOD: Validate pin numbers
if (pin < SOC_GPIO_PIN_COUNT) {
    perimanSetPinBus(pin, ESP32_BUS_TYPE_GPIO, NULL, -1, -1);
}
```

### Issue: Pin Assignment Fails Silently

**Cause**: Not checking return values

**Solution**:
```c
// BAD: Ignoring return value
perimanSetPinBus(pin, type, bus, -1, -1);

// GOOD: Check and handle errors
if (!perimanSetPinBus(pin, type, bus, -1, -1)) {
    log_e("Pin %d assignment failed: %s", pin, perimanGetTypeName(type));
    // Handle error appropriately
}
```

### Issue: Resource Leaks on Pin Switching

**Cause**: Missing or incorrect deinit callbacks

**Solution**:
```c
static bool proper_deinit(void* bus) {
    // Ensure complete cleanup
    if (bus != NULL) {
        // Perform actual hardware deinitialization
        return cleanup_peripheral(bus);
    }
    return true;
}

// Register callback before any pin operations
perimanSetBusDeinit(ESP32_BUS_TYPE_CUSTOM, proper_deinit);
```

## Performance Considerations

### Memory Usage

- Each GPIO pin consumes sizeof(peripheral_pin_item_t) bytes
- Typical memory usage: ~8 bytes per pin
- Total for ESP32 (40 pins): ~320 bytes

### Runtime Performance

- Pin lookups: O(1) - direct array access
- Type name lookups: O(1) - switch statement
- Minimal overhead for most operations

### Optimization Tips

1. **Batch Pin Operations**: Group pin assignments when possible
2. **Minimize Type Queries**: Cache peripheral types if frequently accessed
3. **Use Static Bus Handles**: Avoid dynamic allocation for bus handles

## Compatibility Notes

### ESP32 Variants

The peripheral manager automatically adapts to different ESP32 variants:

- **ESP32**: All peripheral types supported
- **ESP32-S2**: USB-OTG, touch sensor variations
- **ESP32-S3**: Enhanced touch, USB support
- **ESP32-C3**: Reduced peripheral set, no touch sensors
- **ESP32-C6**: 802.15.4, reduced pin count
- **ESP32-H2**: Thread/Zigbee support

### Arduino Core Versions

- **Version 3.0+**: Full peripheral manager support
- **Version 2.x**: Limited or no support
- **Migration**: Update to latest core for full functionality

### IDF Compatibility

The peripheral manager works with:
- ESP-IDF 5.0+
- Arduino Core 3.0+
- Maintains compatibility with existing HAL layers

## Conclusion

The ESP32 Peripheral Manager provides a robust foundation for GPIO pin management in complex applications. By following the patterns and best practices outlined in this documentation, developers can create reliable, conflict-free peripheral management systems.

For additional support:
- Check the ESP32 Arduino Core documentation
- Review example implementations in the Arduino libraries
- Consult the ESP-IDF documentation for low-level details

---

*This documentation was generated from the ESP32 Arduino Core source code analysis. For the most current information, always refer to the official source repository.*