#!/usr/bin/env python3
"""
Simple test MCP server that implements the basic MCP protocol.
This is just for testing the secret management feature.

The server implements JSON-RPC 2.0 over stdin/stdout and provides
a single test tool that demonstrates secret usage.
"""

import json
import os
import sys
from typing import Dict, Any, Optional, List


class TestMCPServer:
    """A simple MCP server for testing secret management functionality."""

    def __init__(self):
        """Initialize the test MCP server."""
        self.protocol_version = "2024-11-05"
        self.server_name = "test-secrets-server"
        self.server_version = "1.0.0"

    def run(self) -> None:
        """Main server loop - read from stdin and process requests."""
        try:
            for line in sys.stdin:
                line = line.strip()
                if not line:
                    continue

                try:
                    request = json.loads(line)
                    self.handle_request(request)
                except json.JSONDecodeError:
                    # Ignore invalid JSON - this is expected behavior for MCP
                    continue
                except Exception as e:
                    # Log errors to stderr (won't interfere with JSON-RPC)
                    print(f"Error processing request: {e}", file=sys.stderr)

        except KeyboardInterrupt:
            # Clean shutdown on Ctrl+C
            pass
        except EOFError:
            # Clean shutdown when stdin is closed
            pass

    def handle_request(self, request: Dict[str, Any]) -> None:
        """Handle an incoming JSON-RPC request."""
        request_id = request.get("id")
        method = request.get("method", "")
        params = request.get("params", {})

        if method == "initialize":
            self.handle_initialize(request_id)
        elif method == "notifications/initialized":
            # No response needed for notifications
            pass
        elif method == "tools/list":
            self.handle_tools_list(request_id)
        elif method == "tools/call":
            self.handle_tools_call(request_id, params)
        else:
            self.send_error(request_id, -32601, f"Method not found: {method}")

    def handle_initialize(self, request_id: Any) -> None:
        """Handle the initialize request."""
        response = {
            "protocolVersion": self.protocol_version,
            "capabilities": {
                "tools": {}
            },
            "serverInfo": {
                "name": self.server_name,
                "version": self.server_version
            }
        }
        self.send_response(request_id, response)

    def handle_tools_list(self, request_id: Any) -> None:
        """Handle the tools/list request."""
        tools = [
            {
                "name": "test_secret",
                "description": "Test tool that demonstrates secret usage",
                "inputSchema": {
                    "type": "object",
                    "properties": {},
                    "required": []
                }
            }
        ]
        self.send_response(request_id, {"tools": tools})

    def handle_tools_call(self, request_id: Any, params: Dict[str, Any]) -> None:
        """Handle the tools/call request."""
        tool_name = params.get("name", "")

        if tool_name == "test_secret":
            # Get the secret from environment variable
            api_key = os.environ.get("TEST_API_KEY", "NO_SECRET_FOUND")

            content = [
                {
                    "type": "text",
                    "text": f"✅ Here's your API Key: {api_key}"
                }
            ]
            self.send_response(request_id, {"content": content})
        else:
            self.send_error(request_id, -32601, f"Tool not found: {tool_name}")

    def send_response(self, request_id: Any, result: Dict[str, Any]) -> None:
        """Send a successful JSON-RPC response."""
        response = {
            "jsonrpc": "2.0",
            "id": request_id,
            "result": result
        }
        print(json.dumps(response), flush=True)

    def send_error(self, request_id: Any, code: int, message: str) -> None:
        """Send an error JSON-RPC response."""
        response = {
            "jsonrpc": "2.0",
            "id": request_id,
            "error": {
                "code": code,
                "message": message
            }
        }
        print(json.dumps(response), flush=True)


def main() -> None:
    """Entry point for the test MCP server."""
    server = TestMCPServer()
    server.run()


if __name__ == "__main__":
    main()
