package io.meshcloud.entitymodel_projection_issue.controller

import io.meshcloud.entitymodel_projection_issue.SimpleRepresentationModelAssembler
import io.meshcloud.entitymodel_projection_issue.model.EntityWithIsFieldProjection
import io.meshcloud.entitymodel_projection_issue.model.EntityWithIsFieldRepository
import io.meshcloud.entitymodel_projection_issue.model.EntityWithoutIsFieldProjection
import io.meshcloud.entitymodel_projection_issue.model.TestEntity
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.PageRequest
import org.springframework.data.projection.ProjectionFactory
import org.springframework.data.web.PagedResourcesAssembler
import org.springframework.hateoas.server.mvc.WebMvcLinkBuilder
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RestController
import kotlin.reflect.jvm.javaMethod

@RestController
class EntityWithIsFieldController {

    @Autowired
    lateinit var repo: EntityWithIsFieldRepository

    @Autowired
    lateinit var projectionFactory: ProjectionFactory

    @Autowired
    lateinit var entityWithIsPageAssembler: PagedResourcesAssembler<EntityWithIsFieldProjection>

    @Autowired
    lateinit var entityWithoutIsPageAssembler: PagedResourcesAssembler<EntityWithoutIsFieldProjection>

    @Autowired
    lateinit var testEntityPageAssembler: PagedResourcesAssembler<TestEntity>

    @GetMapping("/withIs")
    fun findAll(): ResponseEntity<*> {
        val entities = repo.findAll(PageRequest.of(0, 10))

        val projectedEntities = entities.map {
            projectionFactory.createProjection(EntityWithIsFieldProjection::class.java, it)
        }

        val selfLink = WebMvcLinkBuilder.linkTo(this::findAll.javaMethod!!).withSelfRel()

        val pagedEntities = entityWithIsPageAssembler.toModel(
                projectedEntities,
                SimpleRepresentationModelAssembler(),
                selfLink
        )

        return ResponseEntity.ok(pagedEntities)
    }

    @GetMapping("/withoutIs")
    fun findAllWithoutIs(): ResponseEntity<*> {
        val entities = repo.findAll(PageRequest.of(0, 10))

        val projectedEntities = entities.map {
            projectionFactory.createProjection(EntityWithoutIsFieldProjection::class.java, it)
        }

        val selfLink = WebMvcLinkBuilder.linkTo(this::findAll.javaMethod!!).withSelfRel()

        val pagedEntities = entityWithoutIsPageAssembler.toModel(
                projectedEntities,
                SimpleRepresentationModelAssembler(),
                selfLink
        )

        return ResponseEntity.ok(pagedEntities)
    }

    @GetMapping("/entities")
    fun findAllEntities(): ResponseEntity<*> {
        val entities = repo.findAll(PageRequest.of(0, 10))

        val selfLink = WebMvcLinkBuilder.linkTo(this::findAll.javaMethod!!).withSelfRel()

        val pagedEntities = testEntityPageAssembler.toModel(
                entities,
                SimpleRepresentationModelAssembler(),
                selfLink
        )

        return ResponseEntity.ok(pagedEntities)
    }

    @GetMapping("/directProjections")
    fun findDirectProjection(): ResponseEntity<*> {
        val entities = repo.findAll(PageRequest.of(0, 10))

        val projectedEntities = entities.map {
            projectionFactory.createProjection(EntityWithIsFieldProjection::class.java, it)
        }

        return ResponseEntity.ok(projectedEntities)
    }
}