/*! firebase-admin v10.0.1 */
"use strict";
/*!
 * Copyright 2019 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityRules = exports.Ruleset = exports.RulesetMetadataList = void 0;
var validator = require("../utils/validator");
var security_rules_api_client_internal_1 = require("./security-rules-api-client-internal");
var security_rules_internal_1 = require("./security-rules-internal");
/**
 * A page of ruleset metadata.
 */
var RulesetMetadataList = /** @class */ (function () {
    /**
     * @internal
     */
    function RulesetMetadataList(response) {
        if (!validator.isNonNullObject(response) || !validator.isArray(response.rulesets)) {
            throw new security_rules_internal_1.FirebaseSecurityRulesError('invalid-argument', "Invalid ListRulesets response: " + JSON.stringify(response));
        }
        this.rulesets = response.rulesets.map(function (rs) {
            return {
                name: stripProjectIdPrefix(rs.name),
                createTime: new Date(rs.createTime).toUTCString(),
            };
        });
        if (response.nextPageToken) {
            this.nextPageToken = response.nextPageToken;
        }
    }
    return RulesetMetadataList;
}());
exports.RulesetMetadataList = RulesetMetadataList;
/**
 * A set of Firebase security rules.
 */
var Ruleset = /** @class */ (function () {
    /**
     * @internal
     */
    function Ruleset(ruleset) {
        if (!validator.isNonNullObject(ruleset) ||
            !validator.isNonEmptyString(ruleset.name) ||
            !validator.isNonEmptyString(ruleset.createTime) ||
            !validator.isNonNullObject(ruleset.source)) {
            throw new security_rules_internal_1.FirebaseSecurityRulesError('invalid-argument', "Invalid Ruleset response: " + JSON.stringify(ruleset));
        }
        this.name = stripProjectIdPrefix(ruleset.name);
        this.createTime = new Date(ruleset.createTime).toUTCString();
        this.source = ruleset.source.files || [];
    }
    return Ruleset;
}());
exports.Ruleset = Ruleset;
/**
 * The Firebase `SecurityRules` service interface.
 */
var SecurityRules = /** @class */ (function () {
    /**
     * @param app - The app for this SecurityRules service.
     * @constructor
     * @internal
     */
    function SecurityRules(app) {
        this.app = app;
        this.client = new security_rules_api_client_internal_1.SecurityRulesApiClient(app);
    }
    /**
     * Gets the {@link Ruleset} identified by the given
     * name. The input name should be the short name string without the project ID
     * prefix. For example, to retrieve the `projects/project-id/rulesets/my-ruleset`,
     * pass the short name "my-ruleset". Rejects with a `not-found` error if the
     * specified `Ruleset` cannot be found.
     *
     * @param name - Name of the `Ruleset` to retrieve.
     * @returns A promise that fulfills with the specified `Ruleset`.
     */
    SecurityRules.prototype.getRuleset = function (name) {
        return this.client.getRuleset(name)
            .then(function (rulesetResponse) {
            return new Ruleset(rulesetResponse);
        });
    };
    /**
     * Gets the {@link Ruleset} currently applied to
     * Cloud Firestore. Rejects with a `not-found` error if no ruleset is applied
     * on Firestore.
     *
     * @returns A promise that fulfills with the Firestore ruleset.
     */
    SecurityRules.prototype.getFirestoreRuleset = function () {
        return this.getRulesetForRelease(SecurityRules.CLOUD_FIRESTORE);
    };
    /**
     * Creates a new {@link Ruleset} from the given
     * source, and applies it to Cloud Firestore.
     *
     * @param source - Rules source to apply.
     * @returns A promise that fulfills when the ruleset is created and released.
     */
    SecurityRules.prototype.releaseFirestoreRulesetFromSource = function (source) {
        var _this = this;
        return Promise.resolve()
            .then(function () {
            var rulesFile = _this.createRulesFileFromSource('firestore.rules', source);
            return _this.createRuleset(rulesFile);
        })
            .then(function (ruleset) {
            return _this.releaseFirestoreRuleset(ruleset)
                .then(function () {
                return ruleset;
            });
        });
    };
    /**
     * Applies the specified {@link Ruleset} ruleset
     * to Cloud Firestore.
     *
     * @param ruleset - Name of the ruleset to apply or a `RulesetMetadata` object
     *   containing the name.
     * @returns A promise that fulfills when the ruleset is released.
     */
    SecurityRules.prototype.releaseFirestoreRuleset = function (ruleset) {
        return this.releaseRuleset(ruleset, SecurityRules.CLOUD_FIRESTORE);
    };
    /**
     * Gets the {@link Ruleset} currently applied to a
     * Cloud Storage bucket. Rejects with a `not-found` error if no ruleset is applied
     * on the bucket.
     *
     * @param bucket - Optional name of the Cloud Storage bucket to be retrieved. If not
     *   specified, retrieves the ruleset applied on the default bucket configured via
     *   `AppOptions`.
     * @returns A promise that fulfills with the Cloud Storage ruleset.
     */
    SecurityRules.prototype.getStorageRuleset = function (bucket) {
        var _this = this;
        return Promise.resolve()
            .then(function () {
            return _this.getBucketName(bucket);
        })
            .then(function (bucketName) {
            return _this.getRulesetForRelease(SecurityRules.FIREBASE_STORAGE + "/" + bucketName);
        });
    };
    /**
     * Creates a new {@link Ruleset} from the given
     * source, and applies it to a Cloud Storage bucket.
     *
     * @param source - Rules source to apply.
     * @param bucket - Optional name of the Cloud Storage bucket to apply the rules on. If
     *   not specified, applies the ruleset on the default bucket configured via
     *   {@link firebase-admin.app#AppOptions}.
     * @returns A promise that fulfills when the ruleset is created and released.
     */
    SecurityRules.prototype.releaseStorageRulesetFromSource = function (source, bucket) {
        var _this = this;
        return Promise.resolve()
            .then(function () {
            // Bucket name is not required until the last step. But since there's a createRuleset step
            // before then, make sure to run this check and fail early if the bucket name is invalid.
            _this.getBucketName(bucket);
            var rulesFile = _this.createRulesFileFromSource('storage.rules', source);
            return _this.createRuleset(rulesFile);
        })
            .then(function (ruleset) {
            return _this.releaseStorageRuleset(ruleset, bucket)
                .then(function () {
                return ruleset;
            });
        });
    };
    /**
     * Applies the specified {@link Ruleset} ruleset
     * to a Cloud Storage bucket.
     *
     * @param ruleset - Name of the ruleset to apply or a `RulesetMetadata` object
     *   containing the name.
     * @param bucket - Optional name of the Cloud Storage bucket to apply the rules on. If
     *   not specified, applies the ruleset on the default bucket configured via
     *   {@link firebase-admin.app#AppOptions}.
     * @returns A promise that fulfills when the ruleset is released.
     */
    SecurityRules.prototype.releaseStorageRuleset = function (ruleset, bucket) {
        var _this = this;
        return Promise.resolve()
            .then(function () {
            return _this.getBucketName(bucket);
        })
            .then(function (bucketName) {
            return _this.releaseRuleset(ruleset, SecurityRules.FIREBASE_STORAGE + "/" + bucketName);
        });
    };
    /**
     * Creates a {@link RulesFile} with the given name
     * and source. Throws an error if any of the arguments are invalid. This is a local
     * operation, and does not involve any network API calls.
     *
     * @example
     * ```javascript
     * const source = '// Some rules source';
     * const rulesFile = admin.securityRules().createRulesFileFromSource(
     *   'firestore.rules', source);
     * ```
     *
     * @param name - Name to assign to the rules file. This is usually a short file name that
     *   helps identify the file in a ruleset.
     * @param source - Contents of the rules file.
     * @returns A new rules file instance.
     */
    SecurityRules.prototype.createRulesFileFromSource = function (name, source) {
        if (!validator.isNonEmptyString(name)) {
            throw new security_rules_internal_1.FirebaseSecurityRulesError('invalid-argument', 'Name must be a non-empty string.');
        }
        var content;
        if (validator.isNonEmptyString(source)) {
            content = source;
        }
        else if (validator.isBuffer(source)) {
            content = source.toString('utf-8');
        }
        else {
            throw new security_rules_internal_1.FirebaseSecurityRulesError('invalid-argument', 'Source must be a non-empty string or a Buffer.');
        }
        return {
            name: name,
            content: content,
        };
    };
    /**
     * Creates a new {@link Ruleset} from the given {@link RulesFile}.
     *
     * @param file - Rules file to include in the new `Ruleset`.
     * @returns A promise that fulfills with the newly created `Ruleset`.
     */
    SecurityRules.prototype.createRuleset = function (file) {
        var ruleset = {
            source: {
                files: [file],
            },
        };
        return this.client.createRuleset(ruleset)
            .then(function (rulesetResponse) {
            return new Ruleset(rulesetResponse);
        });
    };
    /**
     * Deletes the {@link Ruleset} identified by the given
     * name. The input name should be the short name string without the project ID
     * prefix. For example, to delete the `projects/project-id/rulesets/my-ruleset`,
     * pass the  short name "my-ruleset". Rejects with a `not-found` error if the
     * specified `Ruleset` cannot be found.
     *
     * @param name - Name of the `Ruleset` to delete.
     * @returns A promise that fulfills when the `Ruleset` is deleted.
     */
    SecurityRules.prototype.deleteRuleset = function (name) {
        return this.client.deleteRuleset(name);
    };
    /**
     * Retrieves a page of ruleset metadata.
     *
     * @param pageSize - The page size, 100 if undefined. This is also the maximum allowed
     *   limit.
     * @param nextPageToken - The next page token. If not specified, returns rulesets
     *   starting without any offset.
     * @returns A promise that fulfills with a page of rulesets.
     */
    SecurityRules.prototype.listRulesetMetadata = function (pageSize, nextPageToken) {
        if (pageSize === void 0) { pageSize = 100; }
        return this.client.listRulesets(pageSize, nextPageToken)
            .then(function (response) {
            return new RulesetMetadataList(response);
        });
    };
    SecurityRules.prototype.getRulesetForRelease = function (releaseName) {
        var _this = this;
        return this.client.getRelease(releaseName)
            .then(function (release) {
            var rulesetName = release.rulesetName;
            if (!validator.isNonEmptyString(rulesetName)) {
                throw new security_rules_internal_1.FirebaseSecurityRulesError('not-found', "Ruleset name not found for " + releaseName + ".");
            }
            return _this.getRuleset(stripProjectIdPrefix(rulesetName));
        });
    };
    SecurityRules.prototype.releaseRuleset = function (ruleset, releaseName) {
        if (!validator.isNonEmptyString(ruleset) &&
            (!validator.isNonNullObject(ruleset) || !validator.isNonEmptyString(ruleset.name))) {
            var err = new security_rules_internal_1.FirebaseSecurityRulesError('invalid-argument', 'ruleset must be a non-empty name or a RulesetMetadata object.');
            return Promise.reject(err);
        }
        var rulesetName = validator.isString(ruleset) ? ruleset : ruleset.name;
        return this.client.updateRelease(releaseName, rulesetName)
            .then(function () {
            return;
        });
    };
    SecurityRules.prototype.getBucketName = function (bucket) {
        var bucketName = (typeof bucket !== 'undefined') ? bucket : this.app.options.storageBucket;
        if (!validator.isNonEmptyString(bucketName)) {
            throw new security_rules_internal_1.FirebaseSecurityRulesError('invalid-argument', 'Bucket name not specified or invalid. Specify a default bucket name via the ' +
                'storageBucket option when initializing the app, or specify the bucket name ' +
                'explicitly when calling the rules API.');
        }
        return bucketName;
    };
    SecurityRules.CLOUD_FIRESTORE = 'cloud.firestore';
    SecurityRules.FIREBASE_STORAGE = 'firebase.storage';
    return SecurityRules;
}());
exports.SecurityRules = SecurityRules;
function stripProjectIdPrefix(name) {
    return name.split('/').pop();
}
